function historical_index = import_index_and_tranche_data(sheet, maturity)
% --------------------------------------------------------------------------------------------------
% Imports historical index and tranche prices data for one particular index, like CDX.NA.IG
% --------------------------------------------------------------------------------------------------
% sheet     ... which sheet in the Excel file to import
% maturity  ... time to maturity of index/tranches
% --------------------------------------------------------------------------------------------------
% sample call: import_index_and_tranche_data('5Y IG', 5);
% --------------------------------------------------------------------------------------------------

% Import data
[num, txt] = xlsread('c:/thesis/data/CDX Tranche History.xls', sheet);
dates = datenum(txt(7:end,1));
index_name = txt(7:end,2);

% For each data, determine time to maturity of index
maturities = match_index_maturity_dates(dates, maturity);
T = zeros(length(dates), 1);
for i=1:length(T)
    T(i) = date_diff_years(dates(i), maturities(i));
end
T = min(T, maturity + 0.5);

% Extract index price and up-front payments
index_price = num(:,1);
num_tranches = (size(num,2)-1)/4;
num_dates = size(num, 1);
tranche1_upfront_bid = num(:,2);
tranche1_upfront_ask =  num(:,3);
tranche1_upfront_mid = (tranche1_upfront_bid + tranche1_upfront_ask)/2;
market_upfront_bid = [tranche1_upfront_bid zeros(num_dates, num_tranches-1)];
market_upfront_ask = [tranche1_upfront_ask zeros(num_dates, num_tranches-1)];
market_upfront_mid = [tranche1_upfront_mid zeros(num_dates, num_tranches-1)];

% Extract tranche prices
market_price_bid = [repmat(500, num_dates, 1) num(:, 6:4:(6+4*(num_tranches-2)))];
market_price_ask = [repmat(500, num_dates, 1) num(:, 7:4:(7+4*(num_tranches-2)))];
market_price_mid = (market_price_bid + market_price_ask)/2;

% Extract GC deltas & clean
deltas = num(:, 4:4:(8+4*(num_tranches-2)));
deltas(isnan(deltas)) = 0;

% Generate structure with historical index and tranche prices
historical_index = struct('dates', {dates}, ...
                          'T', T, 'index_market_price', {index_price}, ...
                          'tranche_market_price_bid', {market_price_bid}, ...
                          'tranche_market_price_ask', {market_price_ask}, ...
                          'tranche_market_price_mid', {market_price_mid}, ...
                          'tranche_market_upfront_bid', {market_upfront_bid}, ...
                          'tranche_market_upfront_ask', {market_upfront_ask}, ...
                          'tranche_market_upfront_mid', {market_upfront_mid}, ...
                          'delta_tr_idx_GC', {deltas});                      
